"""
Copyright (C) 2021-2025 NICE s.r.l.
All Rights Reserved

This software is the confidential and proprietary information
of NICE s.r.l. ("Confidential Information").
You shall not disclose such Confidential Information
and shall use it only in accordance with the terms of
the license agreement you entered into with NICE.


Authentication

class:

Authentication -- responsible for getting the authentication token to call the session manager APIs

    methods:

    def _build_client_credentials(client_id, client_secret):
        Encode to Base64 the client id and client secret

    def get_access_token(self, client_id, client_secret, protocol_host_port, no_verify_ssl, ca_bundle):
        Get the access token using the encoded client credentials
"""

import base64
import json

import requests
import urllib3
from typing import Any

from dcvsmcli.auth.errors import CliAuthenticationError
from dcvsmcli.utils import logger
from dcvsmcli.utils.return_codes import CliReturnCodes

log = logger.get()


class Authentication:
    """Authentication class of the CLI"""

    @staticmethod
    def _build_client_credentials(client_id: str, client_secret: str) -> str:
        """Encode to Base64 the client id and client secret

        :param client_id: str
        :param client_secret: str
        :return: encoded client credentials
        """
        client_credentials = "{client_id}:{client_secret}".format(client_id=client_id, client_secret=client_secret)
        return base64.b64encode(client_credentials.encode("utf-8")).decode("utf-8")

    def get_access_token(self, client_id: str, client_secret: str, protocol_host_port: str, no_verify_ssl: bool, ca_bundle: str) -> Any:
        """Get the access token using the encoded client credentials

        :param client_id: str
        :param client_secret: str
        :param protocol_host_port: str
        :param no_verify_ssl: bool
        :param ca_bundle: str
        :return: access token
        """
        log.debug("Getting token from url %s", protocol_host_port)
        client_credentials = self._build_client_credentials(client_id, client_secret)
        headers = {"Authorization": "Basic {}".format(client_credentials), "Content-Type": "application/x-www-form-urlencoded"}

        try:
            if no_verify_ssl:
                urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
                res = requests.post(protocol_host_port, headers=headers, verify=False)
                log.warning("Validation of SSL certificates is disabled")
            else:
                if ca_bundle is None:
                    raise CliAuthenticationError(
                        "The SSL certificate of the broker cannot be verified. Try again using the ca-bundle option.",
                        CliReturnCodes.INVALID_CONFIGURATION,
                    )
                else:
                    res = requests.post(protocol_host_port, headers=headers, verify=ca_bundle)

            if res.status_code == 200:
                log.debug("Response: {}".format(res.text))
                access_token = json.loads(res.text)["access_token"]
                return access_token
            elif res.status_code == 400:
                raise CliAuthenticationError(
                    "Cannot get access token. The provided credentials may not be correct", CliReturnCodes.INVALID_CONFIGURATION
                )
            elif res.status_code == 503:
                raise CliAuthenticationError("The url is not reachable. The broker could not be set properly", CliReturnCodes.SERVICE_ERROR)
            else:
                log.debug(res.text, res.status_code)
                raise CliAuthenticationError("Generic error retrieving access token", CliReturnCodes.SERVICE_ERROR)

        except requests.exceptions.ConnectionError:
            raise CliAuthenticationError("Failed to establish a new connection.", CliReturnCodes.INVALID_CONFIGURATION)
        except (ValueError, KeyError):
            raise CliAuthenticationError("The response is not a valid JSON", CliReturnCodes.SERVICE_ERROR)
