"""
Copyright (C) 2013-2025 NICE s.r.l.
All Rights Reserved

This software is the confidential and proprietary information
of NICE s.r.l. ("Confidential Information").
You shall not disclose such Confidential Information
and shall use it only in accordance with the terms of
the license agreement you entered into with NICE.
"""

import configparser
import os
import platform

import click
from click import pass_obj
from xdg import xdg_config_home

from dcvsmcli._version import __version__
from dcvsmcli.auth.errors import CliAuthenticationError
from dcvsmcli.conf.configuration import CliConfiguration
from dcvsmcli.conf.configuration_parameter import ConfigurationParameter
from dcvsmcli.swagger.swagger import Swagger
from dcvsmcli.swagger_client.rest import ApiException
from dcvsmcli.utils import logger
from dcvsmcli.utils.return_codes import CliReturnCodes
from dcvsmcli.utils.utils import tokenize, validate_multiple_option

log = logger.get()


class CliConfigurationWrapper(object):
    def __init__(self, conf_filename: str, cli_arguments: dict):
        self.conf_filename = conf_filename
        self.cli_arguments = cli_arguments


@click.version_option(prog_name="NICE DCV Session Manager CLI", version=__version__)
@click.group("cli", context_settings=dict(help_option_names=["-h", "--help"]))
@click.option(
    "--conf",
    type=click.STRING,
    default=None,
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="Name of the configuration file to read the configuration parameters",
)
@click.option(
    "--broker-url",
    ConfigurationParameter.BROKER_URL.cli_option_name,
    type=click.STRING,
    default=None,
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="The URL of the broker",
)
@click.option(
    "--debug",
    ConfigurationParameter.DEBUG.cli_option_name,
    type=click.BOOL,
    default=None,
    required=False,
    is_flag=True,
    help="Specify to enable the debug mode. By default is disabled",
)
@click.option(
    "--no-verify-ssl",
    ConfigurationParameter.NO_VERIFY_SSL.cli_option_name,
    type=click.BOOL,
    default=None,
    required=False,
    is_flag=True,
    help="Specify to disable the verification of SSL certification. By default is enabled",
)
@click.option(
    "--output-format",
    ConfigurationParameter.OUTPUT_FORMAT.cli_option_name,
    type=click.STRING,
    default=None,
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="Specify the format of the output",
)
@click.option(
    "--ca-bundle",
    ConfigurationParameter.CA_BUNDLE.cli_option_name,
    type=click.STRING,
    default=None,
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="Specifies the path to a custom certificate bundle (a file with a .pem extension) of CA to use "
    "when establishing SSL/TLS connections",
)
@click.option(
    "--auth-server",
    ConfigurationParameter.AUTH_SERVER.cli_option_name,
    type=click.STRING,
    default=None,
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="URL of the authentication server used to request the token",
)
@click.pass_context
def cli(context: click.Context, conf: str, **arguments):
    """NICE DCV Session Manager CLI"""
    if not conf:
        if platform.system() == "Windows" and os.path.exists("%UserProfile%\\dcvsmcli.conf"):
            conf = "%UserProfile%\\dcvsmcli.conf"
        elif platform.system() == "Linux" and os.path.exists(str(xdg_config_home()) + "/dcvsmcli.conf"):
            conf = str(xdg_config_home()) + "/dcvsmcli.conf"
        else:
            conf = os.path.join(os.path.dirname(__file__), os.pardir, "conf", "dcvsmcli.conf")

    context.obj = CliConfigurationWrapper(conf, arguments)


@cli.command("create-session", short_help="Creates a new NICE DCV session.")
@click.option("--name", type=click.STRING, required=True, multiple=True, callback=validate_multiple_option, help="The name for the session")
@click.option(
    "--owner", type=click.STRING, required=True, multiple=True, callback=validate_multiple_option, help="The name of the session owner"
)
@click.option(
    "--type",
    type=click.Choice(["Console", "Virtual"], case_sensitive=False),
    required=True,
    multiple=True,
    callback=validate_multiple_option,
    help="Session type: CONSOLE|VIRTUAL",
)
@click.option(
    "--init-file",
    type=click.STRING,
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="Supported with virtual sessions on Linux NICE DCV servers",
)
@click.option(
    "--autorun-file",
    type=click.STRING,
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="The path to a file on the host server that is to be run inside the session",
)
@click.option(
    "--autorun-file-arguments",
    type=click.STRING,
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="Command-line arguments passed to AutorunFile upon its execution inside the session",
)
@click.option(
    "--max-concurrent-clients",
    type=click.IntRange(1, 100),
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="The maximum number of concurrent NICE DCV clients, if provided must be between 1 and 100",
)
@click.option(
    "--dcv-gl-enabled",
    type=click.BOOL,
    required=False,
    is_flag=True,
    default=None,
    help="Indicates whether the virtual session is configured to use hardware-based OpenGL. "
    "Specify to enable it, by default is disabled",
)
@click.option(
    "--permissions-file",
    type=click.STRING,
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="The path to the permissions file",
)
@click.option(
    "--requirements",
    type=click.STRING,
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="The requirements that the server must satisfy in order to place the session",
)
@click.option(
    "--storage-root",
    type=click.STRING,
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="Specifies the path to the folder used for session storage",
)
@click.option(
    "--disable-retry-on-failure",
    type=click.BOOL,
    required=False,
    is_flag=True,
    default=None,
    help="Indicates whether to not retry the create session request after it fails on a NICE DCV host for any reason. "
    "By default failed requests are retried",
)
@click.option(
    "--enqueue-request",
    type=click.BOOL,
    required=False,
    is_flag=True,
    default=None,
    help="Indicates whether to queue the request if it can't be immediately fulfilled. " "By default requests are not enqueued",
)
@pass_obj
def create_session(conf_wrapper: CliConfigurationWrapper, **arguments):
    """Creates a new NICE DCV session with the specified details"""
    _get_swagger(conf_wrapper).create_session(
        name=arguments["name"],
        owner=arguments["owner"],
        type=arguments["type"],
        init_file=arguments["init_file"],
        autorun_file=arguments["autorun_file"],
        autorun_file_arguments=tokenize(arguments["autorun_file_arguments"]),
        max_concurrent_clients=arguments["max_concurrent_clients"],
        dcv_gl_enabled=arguments["dcv_gl_enabled"],
        permissions_file=arguments["permissions_file"],
        requirements=arguments["requirements"],
        storage_root=arguments["storage_root"],
        disable_retry_on_failure=arguments["disable_retry_on_failure"],
        enqueue_request=arguments["enqueue_request"],
    )


@cli.command("delete-session", short_help="Deletes the specified NICE DCV session.")
@click.option(
    "--session-id",
    type=click.STRING,
    required=True,
    multiple=True,
    callback=validate_multiple_option,
    help="The ID of the session to delete",
)
@click.option(
    "--owner", type=click.STRING, required=True, multiple=True, callback=validate_multiple_option, help="The owner of the session to delete"
)
@click.option(
    "--force",
    type=click.BOOL,
    required=False,
    is_flag=True,
    default=None,
    help="Specify to remove a session from the Broker's cache with attempting to delete it from the NICE DCV server. "
    "By default is disabled",
)
@pass_obj
def delete_session(conf_wrapper: CliConfigurationWrapper, **arguments):
    """Deletes the specified NICE DCV session and removes it from the Broker's cache"""
    _get_swagger(conf_wrapper).delete_session(session_id=arguments["session_id"], owner=arguments["owner"], force=arguments["force"])


@cli.command("describe-sessions", short_help="Describes one or more NICE DCV sessions.")
@click.option(
    "--session-ids",
    type=click.STRING,
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="Comma separated list of the IDs of the sessions to describe",
)
@click.option(
    "--next-token",
    type=click.STRING,
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="The token to use to retrieve the next page of results",
)
@click.option(
    "--owner",
    type=click.STRING,
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="The owner of the session to describe",
)
@click.option(
    "--max-results",
    type=click.IntRange(1, 1000),
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="Number of results to show, if provided must be between 1 and 1000",
)
@pass_obj
def describe_sessions(conf_wrapper: CliConfigurationWrapper, **arguments):
    """Describes one or more NICE DCV sessions"""
    _get_swagger(conf_wrapper).describe_sessions(
        session_ids=tokenize(arguments["session_ids"]),
        max_results=arguments["max_results"],
        next_token=arguments["next_token"],
        owner=arguments["owner"],
    )


@cli.command("describe-servers", short_help="Describes one or more NICE DCV servers.")
@click.option(
    "--server-ids",
    type=click.STRING,
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="Comma separated list of the IDs of the NICE DCV servers to describe",
)
@click.option(
    "--next-token",
    type=click.STRING,
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="The token to use to retrieve the next page of results",
)
@click.option(
    "--max-results",
    type=click.IntRange(1, 1000),
    required=False,
    multiple=True,
    callback=validate_multiple_option,
    help="The maximum number of results to be returned by the request in paginated output, " "if provided must be between 1 and 1000",
)
@pass_obj
def describe_servers(conf_wrapper: CliConfigurationWrapper, **arguments):
    """Describes one or more NICE DCV servers"""
    _get_swagger(conf_wrapper).describe_servers(
        server_ids=tokenize(arguments["server_ids"]), next_token=arguments["next_token"], max_results=arguments["max_results"]
    )


@cli.command("open-servers")
@click.option(
    "--server-ids",
    type=click.STRING,
    required=True,
    multiple=True,
    callback=validate_multiple_option,
    help="Comma separated list of the IDs of the NICE DCV servers to open",
)
@pass_obj
def open_servers(conf_wrapper: CliConfigurationWrapper, **arguments):
    """Opens one or more NICE DCV servers. Before you can create NICE DCV sessions on a NICE DCV server, you must change the
    server's state to open. After the NICE DCV server is open, you can create NICE DCV sessions on the server"""
    _get_swagger(conf_wrapper).open_servers(server_ids=tokenize(arguments["server_ids"]))


@cli.command("close-servers")
@click.option(
    "--server-ids",
    type=click.STRING,
    required=True,
    multiple=True,
    callback=validate_multiple_option,
    help="Comma separated list of the ID of the NICE DCV servers to close",
)
@click.option(
    "--force",
    type=click.BOOL,
    required=False,
    is_flag=True,
    default=None,
    help="Specify to force the close operation. By default is disabled",
)
@pass_obj
def close_servers(conf_wrapper: CliConfigurationWrapper, **arguments):
    """Closes one or more NICE DCV servers. When you close a NICE DCV server, you make it unavailable for NICE DCV session
    placement. You cannot create NICE DCV sessions on closed servers. Closing a server ensures that no sessions are
    running on it and that users cannot create new sessions on it"""
    _get_swagger(conf_wrapper).close_servers(server_ids=tokenize(arguments["server_ids"]), force=arguments["force"])


@cli.command("get-session-screenshots", short_help="Gets screenshots of one or more NICE DCV sessions.")
@click.option(
    "--session-ids",
    type=click.STRING,
    required=True,
    multiple=True,
    callback=validate_multiple_option,
    help="Coma separated list of the IDs of the NICE DCV sessions from which to get the screenshots",
)
@click.option("--output-folder", type=click.STRING, required=False, help="The absolute path of the folder where to save the screenshots")
@click.option(
    "--max-width",
    type=click.IntRange(min=1),
    required=False,
    multiple=False,
    help="Specifies the maximum width, in pixels, of session screenshots",
)
@click.option(
    "--max-height",
    type=click.IntRange(min=1),
    required=False,
    multiple=False,
    help="Specifies the maximum height, in pixels, of session screenshots",
)
@pass_obj
def get_session_screenshots(conf_wrapper: CliConfigurationWrapper, **arguments):
    """Gets screenshots of one or more NICE DCV session"""
    _get_swagger(conf_wrapper).get_session_screenshots(
        session_ids=tokenize(arguments["session_ids"]),
        folder=arguments["output_folder"],
        max_width=arguments["max_width"],
        max_height=arguments["max_height"],
    )


@cli.command("get-session-connection-data", short_help="Gets connection data for a specific NICE DCV session.")
@click.option(
    "--session-id",
    type=click.STRING,
    required=True,
    multiple=True,
    callback=validate_multiple_option,
    help="The ID of the session for which to view connection information",
)
@click.option(
    "--user",
    type=click.STRING,
    required=True,
    multiple=True,
    callback=validate_multiple_option,
    help="The name of the user for which to view connection information",
)
@pass_obj
def get_session_connection_data(conf_wrapper: CliConfigurationWrapper, **arguments):
    """Gets connection information for a specific user's connection to a specific NICE DCV session"""
    _get_swagger(conf_wrapper).get_session_connection_data(session_id=arguments["session_id"], user=arguments["user"])


@cli.command("update-session-permissions", short_help="Updates the user permissions for a specific NICE DCV session.")
@click.option(
    "--session-id",
    type=click.STRING,
    required=True,
    multiple=True,
    callback=validate_multiple_option,
    help="The ID of the session for which to change the permissions",
)
@click.option(
    "--owner",
    type=click.STRING,
    required=True,
    multiple=True,
    callback=validate_multiple_option,
    help="The owner of the session for which to change the permissions",
)
@click.option(
    "--permission-file",
    type=click.STRING,
    required=True,
    multiple=True,
    callback=validate_multiple_option,
    help="The path to the permission file",
)
@pass_obj
def update_session_permissions(conf_wrapper: CliConfigurationWrapper, **arguments):
    """Updates the user permissions for a specific NICE DCV session"""
    _get_swagger(conf_wrapper).update_session_permissions(
        session_id=arguments["session_id"], owner=arguments["owner"], permissions_file=arguments["permission_file"]
    )


def _get_configuration(conf_filename: str, arguments: dict) -> CliConfiguration:
    conf_parser = configparser.ConfigParser()
    conf_parser.read_file(open(conf_filename, "r"))
    configuration = CliConfiguration()
    configuration.load(conf_parser, arguments)
    return configuration


def _get_swagger(conf_wrapper: CliConfigurationWrapper) -> Swagger:
    configuration = _get_configuration(conf_wrapper.conf_filename, conf_wrapper.cli_arguments)

    return Swagger(
        configuration.get(ConfigurationParameter.BROKER_URL),
        configuration.get(ConfigurationParameter.OAUTH2_TOKEN),
        configuration.get(ConfigurationParameter.NO_VERIFY_SSL),
        configuration.get(ConfigurationParameter.CA_BUNDLE),
    )


def run():
    try:
        cli()
    except OSError as error:
        log.error("Error reading file or directory. {}".format(str(error)))
        return CliReturnCodes.COMMAND_SYNTAX_INVALID
    except configparser.Error as error:
        log.error("An error occurred while parsing the configuration file. {}".format(str(error)))
        return CliReturnCodes.CONFIG_FILE_ERROR
    except CliAuthenticationError as error:
        log.error(error.message)
        return error.return_code
    except (ValueError, TypeError) as error:
        log.error("Error reading configuration. {}".format(error))
        return CliReturnCodes.COMMAND_SYNTAX_INVALID
    except ApiException as error:
        log.error("Error calling the broker endpoint. {}".format(str(error)))
        return CliReturnCodes.SERVICE_ERROR
    except KeyboardInterrupt:
        log.error("The process received a SIGINT (Ctrl-C)")
        return CliReturnCodes.SIGINT_SIGNAL
    except Exception as e:
        log.error("Unexpected error occurred, see traceback below")
        log.exception(e)
        return CliReturnCodes.GENERAL_ERROR

    return CliReturnCodes.SUCCESSFUL
