"""
Copyright (C) 2021-2025 NICE s.r.l.
All Rights Reserved

This software is the confidential and proprietary information
of NICE s.r.l. ("Confidential Information").
You shall not disclose such Confidential Information
and shall use it only in accordance with the terms of
the license agreement you entered into with NICE.


CLI configuration.

The CLI will read all the configuration parameters from the environment variables and the configuration file

class:

CliConfiguration -- responsible for loading all the configuration parameters


    methods:

    def load(config, cli_configuration)
        Load all the configuration parameters
        Return the instance of the class

    def _get_from_conf_file(config, parameter)
        Read the parameter from the configuration file
        If the configuration file has any section not well formatted it will raise NoSectionError exception and exit with
        code 1
        If the configuration file is missing an option it will raise NoOptionError exception and return None

    def _get_from_env_var(parameter):
        Read the parameter from the environment variables

    def _set_auth_server(self):
        Set the default value of authentication server parameter

    def get_default_value(parameter)
        Return the default value of the configuration parameter

    def get(self, parameter):
        Return the value of the parameter

    def _put_if_some(self, parameter, value):
        Set the parameter value in the _configuration dict only if value is not None, empty string and correct type
"""

import os
from configparser import ConfigParser, NoOptionError, NoSectionError
from typing import Any, Optional
from urllib.parse import urljoin

from dcvsmcli.auth.authentication import Authentication
from dcvsmcli.conf import validation
from dcvsmcli.conf.configuration_parameter import ConfigurationParameter
from dcvsmcli.utils import logger

log = logger.get()
TO_REDACT = [ConfigurationParameter.OAUTH2_TOKEN, ConfigurationParameter.CLIENT_PASSWORD]


class CliConfiguration:
    """Configuration class of the CLI"""

    def __init__(self):
        self._default = {
            ConfigurationParameter.OUTPUT_FORMAT: "json",
            ConfigurationParameter.DEBUG: "false",
            ConfigurationParameter.NO_VERIFY_SSL: "false",
        }
        self._configuration = {}
        for parameter in ConfigurationParameter:
            self._configuration[parameter] = None

    def load(self, config: ConfigParser, arguments: dict):
        """Load all the configuration parameters following the priorities:
        1 - Read the parameter value from command line
        2 - Read the parameter value from environment variables
        3 - Read the parameter value from configuration file
        4 - Read the parameter default value from the _default dict

        :param config: ConfigParser
        :param arguments: dictionary
        :return: instance of the class
        """

        # Reading log level before other configs to be able to log correctly
        self._read_debug_level(config, arguments)

        # Read parameters default value
        for parameter in self._configuration.keys():
            self._put_if_some(parameter, self.get_default(parameter))

        # Read parameters value from configuration file and environment variables
        for parameter in ConfigurationParameter:
            self._put_if_some(parameter, self._get_from_conf_file(config, parameter))
            self._put_if_some(parameter, self._get_from_env_var(parameter))

        # Read parameters value from CLI
        for cli_option, cli_value in arguments.items():
            if cli_value is not None:
                for parameter, value in self._configuration.items():
                    if parameter.cli_option_name is cli_option:
                        self._put_if_some(parameter, str(cli_value))

        if self._configuration[ConfigurationParameter.AUTH_SERVER] is None:
            self._set_auth_server()

        if self._configuration[ConfigurationParameter.OAUTH2_TOKEN] is None:
            auth = Authentication()
            oauth2_token = auth.get_access_token(
                self.get(ConfigurationParameter.CLIENT_ID),
                self.get(ConfigurationParameter.CLIENT_PASSWORD),
                self.get(ConfigurationParameter.AUTH_SERVER),
                self.get(ConfigurationParameter.NO_VERIFY_SSL),
                self.get(ConfigurationParameter.CA_BUNDLE),
            )
            self._put_if_some(ConfigurationParameter.OAUTH2_TOKEN, oauth2_token)
        self._log_configuration_map()
        return self

    def _log_configuration_map(self):
        log.debug("Configuration loaded.")
        for key in self._configuration.keys():
            if key in TO_REDACT and self._configuration[key] is not None:
                value = "**********"
            else:
                value = self._configuration[key]
            log.debug("{} : {}".format(key, value))

    def _read_debug_level(self, config, arguments):
        debug = ConfigurationParameter.DEBUG

        self._put_if_some(debug, self.get_default(debug))
        self._put_if_some(debug, self._get_from_conf_file(config, debug))
        if arguments.get(debug.cli_option_name, None) is not None:
            self._put_if_some(debug, str(arguments[debug.cli_option_name]))
        if self._configuration[debug]:
            logger.set_debug_level()

    @staticmethod
    def _get_from_conf_file(config: ConfigParser, parameter: ConfigurationParameter) -> Any:
        """Read the parameter from the configuration file

        If the configuration file has any section not well formatted it will raise NoSectionError exception and exit with
        code 1
        If the configuration file is missing an option it will raise NoOptionError exception and return None

        :param config: ConfigParser
        :param parameter: ConfigurationParameter
        :return: Any
        """
        if parameter.conf_section_name:
            try:
                return config.get(parameter.conf_section_name, parameter.conf_param_name)
            except NoSectionError:
                log.debug("Section: '{}' was not found in the configuration file".format(parameter.conf_section_name))
            except NoOptionError:
                log.debug(
                    "Section '{}' does not contain the parameter: '{}'".format(parameter.conf_section_name, parameter.conf_param_name)
                )
            return None

    @staticmethod
    def _get_from_env_var(parameter: ConfigurationParameter) -> Optional[str]:
        """Read the parameter from the environment variables

        :param parameter: ConfigurationParameter
        :return: Optional[str]
        """

        if parameter.env_var_name:
            return os.environ.get(parameter.env_var_name, None)

        return None

    def _set_auth_server(self) -> Any:
        """Set the default value of authentication server parameter

        :return: Any
        """
        broker_url = self._configuration[ConfigurationParameter.BROKER_URL]
        if broker_url:
            self._configuration[ConfigurationParameter.AUTH_SERVER] = urljoin(broker_url, "oauth2/token?grant_type=client_credentials")
        else:
            log.warning("The broker url is not defined, cannot retrieve default authorization server")
            return None

    def get_default(self, parameter: ConfigurationParameter) -> Any:
        """Return the default value of the parameter

        :param parameter: ConfigurationParameter
        :return: Any
        """
        return self._default.get(parameter, None)

    def get(self, parameter: ConfigurationParameter) -> Any:
        """Return the value of the parameter

        :param parameter: ConfigurationParameter
        :return: Any
        """
        return self._configuration[parameter]

    def _put_if_some(self, parameter: ConfigurationParameter, value: Optional[str]):
        """Set the parameter value in the _configuration dict only if value is not None, empty string and correct type

        :param parameter: ConfigurationParameter
        :param value: Optional[str]
        :return: None
        """

        if validation.to_string_value(value):
            self._configuration[parameter] = parameter.to_value(value)
