"""
Copyright (C) 2021-2025 NICE s.r.l.
All Rights Reserved

This software is the confidential and proprietary information
of NICE s.r.l. ("Confidential Information").
You shall not disclose such Confidential Information
and shall use it only in accordance with the terms of
the license agreement you entered into with NICE.


Validation

The validation module is used to validate and eventually sanitize the value of a configuration parameters.
Once validated, the input value, which is always a string coming from the command line, is converted
to the appropriate type before being returned to the caller.

methods:

    validate(param_name: str, func: callable) -> Any:
        Call the nested function to identify which parameter has an invalid value

    to_uri_value(param_name: str, value: str) -> str:
        Validate if the url parsed is valid
        If present remove multiple slashes from the original url using urljoin

    to_boolean_value(param_name: str, value: str) -> bool:
        Convert the string to boolean value

    to_output_format_value(value: str) -> str:
        Verify if the output format specified by the user is supported

    to_string_value(value: str) -> str:
        Return the string value stripped and without surrounding decorators such as quotes
"""

from typing import Any, Callable
from urllib.parse import urlparse
from dcvsmcli.utils import utils


def validate(param_name: str, func: Callable) -> Any:
    """
    :param param_name: str
    :param func: Callable
    :return: Any
    """

    def validate_val(value: str) -> Any:
        """
        :param value: str
        :return: Any
        """
        return func(param_name, value)

    return validate_val


def to_uri_value(param_name: str, value: str) -> Any:
    """
    :param param_name: str
    :param value: str
    :return: Any
    """

    url = to_string_value(value)
    if url:
        uri = urlparse(url)
        valid = all([uri.scheme, uri.netloc])
        if valid:
            return url
        else:
            raise ValueError(
                "The value of parameter '{}' is not valid: '{}'. "
                "\nExample of valid URL: 'https://example.com:8443'".format(param_name, value)
            )
    else:
        raise TypeError("The value of parameter '{}' cannot be None or empty".format(param_name))


def to_boolean_value(param_name: str, value: str) -> bool:
    """
    :param param_name: str
    :param value: str
    :return: bool
    """

    val = to_string_value(value)
    if val:
        val_lower = val.lower()
        if val_lower == "true":
            return True
        elif val_lower == "false":
            return False
        else:
            raise ValueError("The value '{}' of parameter '{}' is not boolean".format(val, param_name))
    else:
        raise TypeError("The value of parameter '{}' cannot be None or empty".format(param_name))


def to_output_format_value(value: str) -> Any:
    """
    :param value: str
    :return: Any
    """

    supported_formats = ["json"]

    val = to_string_value(value)
    if val:
        if val in supported_formats:
            return val
        else:
            raise ValueError("The output format specified '{}' is not supported. Supported formats: {}".format(val, supported_formats))
    else:
        raise TypeError("The output format cannot be None or empty")


def to_string_value(value: str) -> Any:
    """
    :param value: str
    :return: Any
    """
    if utils.is_not_blank(value):
        for decorator in ("'", '"'):
            s, decorated = _strip_string_decorators(value, decorator)
            if decorated:
                return s.strip()

        return value.strip()

    return None


def _strip_string_decorators(value: str, decorator: str) -> tuple:
    """
    :param value: str
    :param decorator: str
    :return: tuple
    """
    if value.startswith(decorator):
        if value.endswith(decorator):
            return value[len(decorator) : -len(decorator)], True
        else:
            raise ValueError("Closing {} is missing from string {}".format(decorator, value))

    if value.endswith(decorator):
        raise ValueError("Opening {} is missing from string {}".format(decorator, value))

    return value, False
