"""
Copyright (C) 2021-2025 NICE s.r.l.
All Rights Reserved

This software is the confidential and proprietary information
of NICE s.r.l. ("Confidential Information").
You shall not disclose such Confidential Information
and shall use it only in accordance with the terms of
the license agreement you entered into with NICE.


Swagger

class:

Swagger -- implementation of the functions to call the APIs of the session manager

    methods:

    def _set_request_headers(self, api_client):
        Set the headers to the api client

    def _get_sessions_api(self):
        Return the api instance of the sessions

    def _get_servers_api(self):
        Return the api instance of the server

    def _get_client_configuration(self):
        Return the configuration of the client

    def _get_session_connection_api(self):
        Return the api instance of the session connection data

    def _get_session_permissions_api(self):
        Return the api instance of the session permissions

    def create_sessions(self,
                        name: str,
                        owner: str,
                        type: str,
                        init_file: str,
                        autorun_file: str,
                        autorun_file_arguments: str,
                        max_concurrent_clients: int,
                        dcv_gl_enabled: bool,
                        permissions_file: str,
                        requirements: str,
                        storage_root: str,
                        disable_retry_on_failure: bool,
                        enqueue_request: bool):
        Creates a new NICE DCV session with the specified details

    def describe_sessions(self, session_ids=None, next_token=None, tags=None, owner=None):
        Describes one or more NICE DCV sessions

    def delete_sessions(self, sessions_to_delete, force=False):
        Deletes the specified NICE DCV session

    def open_servers(self, server_ids):
        Opens one or more NICE DCV servers

    def describe_servers(self, server_ids=None, next_token=None, max_results=None):
        Describes one or more NICE DCV servers.

    def close_servers(self, server_ids):
        Closes one or more NICE DCV servers

    def get_session_connection_data(self, session_id, user):
        Gets connection information for a specific user's connection to a specific NICE DCV session.

    def get_session_screenshots(self, session_ids, max_width, max_height):
        Gets screenshots of one or more NICE DCV sessions. Parameters max_width and max_height can be null.

    def update_session_permissions(self, session_permissions_to_update):
        Updates the user permissions for a specific NICE DCV session
"""

# APIs
from dcvsmcli.swagger_client.api.get_session_connection_data_api import GetSessionConnectionDataApi
from dcvsmcli.swagger_client.api.servers_api import ServersApi
from dcvsmcli.swagger_client.api.session_permissions_api import SessionPermissionsApi
from dcvsmcli.swagger_client.api.sessions_api import SessionsApi

# ApiClient
from dcvsmcli.swagger_client.api_client import ApiClient
from dcvsmcli.swagger_client.configuration import Configuration

# Models
from dcvsmcli.swagger_client.models import OpenServerRequestData, CloseServerRequestData
from dcvsmcli.swagger_client.models.create_session_request_data import CreateSessionRequestData
from dcvsmcli.swagger_client.models.delete_session_request_data import DeleteSessionRequestData
from dcvsmcli.swagger_client.models.describe_servers_request_data import DescribeServersRequestData
from dcvsmcli.swagger_client.models.describe_sessions_request_data import DescribeSessionsRequestData
from dcvsmcli.swagger_client.models.get_session_screenshot_request_data import GetSessionScreenshotRequestData
from dcvsmcli.swagger_client.models.key_value_pair import KeyValuePair
from dcvsmcli.swagger_client.models.update_session_permissions_request_data import UpdateSessionPermissionsRequestData

from dcvsmcli.utils import json_utils, logger, utils

from typing import Optional

log = logger.get()


class Swagger:
    """Swagger class of the CLI"""

    def __init__(self, protocol_host_port: str, oauth2_token: str, no_verify_ssl: bool, ca_bundle: str):
        """
        :param protocol_host_port: str
        :param oauth2_token: str
        :param no_verify_ssl: bool
        :param ca_bundle: str
        """
        self.protocol_host_port = protocol_host_port
        self.oauth2_token = oauth2_token
        self.no_verify_ssl = no_verify_ssl
        self.ca_bundle = ca_bundle

    def _set_request_headers(self, api_client: ApiClient):
        """Set the headers to the api client

        :param api_client: ApiClient
        """
        access_token = self.oauth2_token
        api_client.set_default_header(header_name="Authorization", header_value="Bearer {}".format(access_token))

    def _get_sessions_api(self) -> SessionsApi:
        """Return the api instance of the sessions

        :return: SessionsApi
        """
        api_instance = SessionsApi(ApiClient(self._get_client_configuration()))
        self._set_request_headers(api_instance.api_client)
        return api_instance

    def _get_servers_api(self) -> ServersApi:
        """Return the api instance of the server

        :return: ServersApi
        """
        api_instance = ServersApi(ApiClient(self._get_client_configuration()))
        self._set_request_headers(api_instance.api_client)
        return api_instance

    def _get_client_configuration(self) -> Configuration:
        """Return the configuration of the client

        :return: Configuration
        """
        configuration = Configuration()
        configuration.host = self.protocol_host_port
        if self.no_verify_ssl:
            configuration.verify_ssl = False
        else:
            configuration.verify_ssl = True
            configuration.ssl_ca_cert = self.ca_bundle
        log.debug("About to make API call to host %s", configuration.host)
        return configuration

    def _get_session_connection_api(self) -> GetSessionConnectionDataApi:
        """Return the api instance of the session connection data

        :return: GetSessionConnectionDataApi
        """
        api_instance = GetSessionConnectionDataApi(ApiClient(self._get_client_configuration()))
        self._set_request_headers(api_instance.api_client)
        return api_instance

    def _get_session_permissions_api(self) -> SessionPermissionsApi:
        """Return the api instance of the session permissions

        :return: SessionPermissionsApi
        """
        api_instance = SessionPermissionsApi(ApiClient(self._get_client_configuration()))
        self._set_request_headers(api_instance.api_client)
        return api_instance

    def create_session(
        self,
        name: str,
        owner: str,
        type: str,
        init_file: Optional[str] = None,
        autorun_file: Optional[str] = None,
        autorun_file_arguments: Optional[list] = None,
        max_concurrent_clients: Optional[int] = None,
        dcv_gl_enabled: Optional[bool] = None,
        permissions_file: Optional[str] = None,
        requirements: Optional[str] = None,
        storage_root: Optional[str] = None,
        disable_retry_on_failure: Optional[bool] = None,
        enqueue_request: Optional[bool] = None,
    ):
        """Creates a new NICE DCV session with the specified details

        :param name: str
        :param owner: str
        :param type: str
        :param init_file: Optional[str]
        :param autorun_file: Optional[list]
        :param autorun_file_arguments: Optional[str]
        :param max_concurrent_clients: Optional[int]
        :param dcv_gl_enabled: Optional[bool]
        :param permissions_file: Optional[str]
        :param requirements: Optional[str]
        :param storage_root: Optional[str]
        :param disable_retry_on_failure: Optional[bool]
        :param enqueue_request: Optional[bool]
        """
        create_sessions_request = list()
        if permissions_file:
            permissions_file = utils.base64_encode_file(permissions_file)
        a_request = CreateSessionRequestData(
            name=name,
            owner=owner,
            type=type,
            init_file=init_file,
            autorun_file=autorun_file,
            autorun_file_arguments=autorun_file_arguments,
            max_concurrent_clients=max_concurrent_clients,
            dcv_gl_enabled=dcv_gl_enabled,
            permissions_file=permissions_file,
            requirements=requirements,
            storage_root=storage_root,
            disable_retry_on_failure=disable_retry_on_failure,
            enqueue_request=enqueue_request,
        )
        create_sessions_request.append(a_request)

        api_instance = self._get_sessions_api()
        log.debug("Create Sessions Request: {}".format(create_sessions_request))
        api_response = api_instance.create_sessions(body=create_sessions_request)
        json_utils.print_json(api_response)

    def describe_sessions(
        self,
        session_ids: Optional[list] = None,
        max_results: Optional[int] = None,
        next_token: Optional[str] = None,
        tags: Optional[list] = None,
        owner: Optional[str] = None,
    ):
        """Describes one or more NICE DCV sessions

        :param max_results: Optional[int]
        :param session_ids: Optional[list]
        :param next_token: Optional[str]
        :param tags: Optional[list]
        :param owner: Optional[str]
        """
        filters = list()
        if tags:
            for tag in tags:
                filter_key_value_pair = KeyValuePair(key="tag:" + tag["Key"], value=tag["Value"])
                filters.append(filter_key_value_pair)
        if owner:
            filter_key_value_pair = KeyValuePair(key="owner", value=owner)
            filters.append(filter_key_value_pair)

        request = DescribeSessionsRequestData(session_ids=session_ids, filters=filters, max_results=max_results, next_token=next_token)
        log.debug("Describe Sessions Request: {}".format(request))
        api_instance = self._get_sessions_api()
        api_response = api_instance.describe_sessions(body=request)
        json_utils.print_json(api_response)

    def delete_session(self, session_id: str, owner: str, force: Optional[bool] = None):
        """Deletes the specified NICE DCV sessions

        :param session_id: list
        :param owner str
        :param force: Optional[bool]
        """
        delete_sessions_request = [DeleteSessionRequestData(session_id=session_id, owner=owner, force=force)]

        log.debug("Delete Sessions Request: {}".format(delete_sessions_request))
        api_instance = self._get_sessions_api()
        api_response = api_instance.delete_sessions(body=delete_sessions_request)
        json_utils.print_json(api_response)

    def open_servers(self, server_ids: list):
        """Opens one or more NICE DCV servers

        :param server_ids: list
        """
        request = [OpenServerRequestData(server_id=server_id) for server_id in server_ids]
        log.debug("Open Servers Request: {}".format(request))
        api_instance = self._get_servers_api()
        api_response = api_instance.open_servers(body=request)
        json_utils.print_json(api_response)

    def describe_servers(self, server_ids=None, next_token=None, max_results=None):
        """Describes one or more NICE DCV servers

        :param server_ids: list
        :param next_token: str
        :param max_results: int
        """
        request = DescribeServersRequestData(server_ids=server_ids, next_token=next_token, max_results=max_results)
        log.debug("Describe Servers Request: {}".format(request))
        api_instance = self._get_servers_api()
        api_response = api_instance.describe_servers(body=request)
        json_utils.print_json(api_response)

    def close_servers(self, server_ids: list, force: Optional[bool] = None):
        """Closes one or more NICE DCV servers

        :param force: bool
        :param server_ids: list
        :param force: Optional[bool]
        """
        request = [CloseServerRequestData(server_id=server_id, force=force) for server_id in server_ids]
        log.debug("Close Servers Request: {}".format(request))
        api_instance = self._get_servers_api()
        api_response = api_instance.close_servers(body=request)
        json_utils.print_json(api_response)

    def get_session_connection_data(self, session_id: str, user: str):
        """Gets connection information for a specific user's connection to a specific NICE DCV session

        :param session_id: str
        :param user: str
        """
        api_response = self._get_session_connection_api().get_session_connection_data(session_id=session_id, user=user)
        json_utils.print_json(api_response)

    def get_session_screenshots(self, session_ids: list, folder: str, max_width: Optional[int] = None, max_height: Optional[int] = None):
        """Gets screenshots of one or more NICE DCV sessions

        :param session_ids: list
        :param folder: str
        :param max_width: Optional[int]
        :param max_height: Optional[int]
        """
        request = [
            GetSessionScreenshotRequestData(session_id=session_id, max_width=max_width, max_height=max_height) for session_id in session_ids
        ]

        log.debug("Get Session Screenshots Request: {}".format(request))
        api_instance = self._get_sessions_api()
        api_response = api_instance.get_session_screenshots(body=request)
        json_utils.print_json(api_response)
        if folder:
            utils.save_screenshots(api_response.successful_list, folder)

    def update_session_permissions(self, session_id: str, owner: str, permissions_file: str):
        """Updates the user permissions for a specific NICE DCV session

        :param session_id: str
        :param owner: str
        :param permissions_file: str
        """
        update_session_permissions_request = list()
        permissions_file = utils.base64_encode_file(permissions_file)
        a_request = UpdateSessionPermissionsRequestData(session_id=session_id, owner=owner, permissions_file=permissions_file)
        update_session_permissions_request.append(a_request)
        log.debug("Update Session Permissions Request: {}".format(update_session_permissions_request))
        api_instance = self._get_session_permissions_api()
        api_response = api_instance.update_session_permissions(body=update_session_permissions_request)
        json_utils.print_json(api_response)
