"""
Copyright (C) 2013-2025 NICE s.r.l.
All Rights Reserved

This software is the confidential and proprietary information
of NICE s.r.l. ("Confidential Information").
You shall not disclose such Confidential Information
and shall use it only in accordance with the terms of
the license agreement you entered into with NICE.


Utils

The utils module contains various utility functions used throughout the whole CLI code

methods:

    is_not_blank(value: str) -> bool:
        Verify if the value contains blank spaces

    base64_encode_file(file: str) -> str:
        Return the base64 encoded file

    save_screenshots(response: dict, folder: str):
        Save the screenshots of the session in the specified absolute path

    validate_multiple_option(ctx: click.Context, param: click.Option, value: Any) -> Any:
        Checks if a Click multiple option value contains more than one item and raise an error otherwise
        If ok, the value is extracted from the tuple and returned

    def tokenize(csl: Optional[str]) -> list:
        Split a comma separated string and return each token in a list
        Comma escaping is supported
"""

import base64
import os
from typing import Any
from typing import Optional

import click

from dcvsmcli.utils.constants import Constants
from dcvsmcli.utils import logger

log = logger.get()


def is_not_blank(value: str) -> bool:
    """
    :param value: str
    :return: bool
    """
    if value and value.strip():
        return True

    return False


def validate_multiple_option(ctx: click.Context, param: click.Option, value: Any) -> Any:
    if not isinstance(value, tuple):
        return value

    if len(value) == 0:
        return None
    if len(value) > 1:
        raise click.UsageError("Parameter '{}' cannot be used multiple times".format(param.opts[0]))
    return value[0]


def tokenize(csl: Optional[str]) -> list:
    tokens = []

    if csl:
        token = None
        parts = csl.split(Constants.LIST_SEPARATOR)
        while parts:
            if not token:
                token = parts.pop(0)
            if token.endswith(Constants.ESCAPE_CHAR) and len(parts) > 0:
                token = Constants.LIST_SEPARATOR.join((token[: -len(Constants.ESCAPE_CHAR)], parts.pop(0)))
            else:
                (lambda t: tokens.append(t) if t else None)(token.strip())
                token = None

        if token:
            (lambda t: tokens.append(t) if t else None)(token.strip())

    return tokens


def base64_encode_file(file: str) -> str:
    """
    :param file: str
    :return: string
    """
    with open(file, "rb") as file_to_encode:
        encoded_file = base64.b64encode(file_to_encode.read()).decode("utf-8")
        return encoded_file


def save_screenshots(successful_responses: dict, folder: str):
    """
    :param successful_responses: dict
    :param folder: str
    """
    display_counter = 1

    for response in successful_responses:
        session_id = response.session_screenshot.session_id
        for image in response.session_screenshot.images:
            encoded_image = image.data
            image_format = image.format
            creation_time = image.creation_time.astimezone().replace(microsecond=0).isoformat().replace(":", ".")
            primary = image.primary
            image = base64.b64decode(encoded_image)

            if primary:
                primary_tag = "P"
            else:
                primary_tag = ""

            file_to_save = "{}{}screenshot_{}_{}{}_{}.{}".format(
                folder.rstrip(os.sep), os.sep, session_id, primary_tag, display_counter, creation_time, image_format
            )
            display_counter += 1

            with open(file_to_save, "wb") as screenshot:
                screenshot.write(image)
                log.info("Screenshot saved successfully: {}".format(file_to_save))
